rem Image reconstruction program Imagcon
rem Reads normalised ASCII capacitance data files and ASCII transpose sensitivity matrix
rem and constructs and plots permittivity images for 8-electrode single-pane sensor data.
rem maxframes is maximum number of capacitance data frames to be input
rem E is the number of electrodes used to capture the data
rem M is number of unique capacitance measurements eg 28
rem p is the number of horizontal and vertical pixels in the grid (=32)
rem CP is the number of pixels inside the circular boundary (=812)
rem P (n) is the total number of pixels in the 32 x 32 pixel grid = P (1024)
rem i is the pixel number in the horizontal direction#
rem j is the pixel number in the vertical direction
rem geom is a p x p (32 x 32) element array which defines the location of active pixels inside a nominal circular vessel
rem geom2 is a P (1024) element array derived from geom which also defines the location of the active pixels
rem C1(m) is the set of normalised plane 1 measured capacitances
rem C2(m) is the set of normalised plane 1 measured capacitances
rem K1(n) is the set of normalised plane 1 pixel permittivity values
rem K2(n) is the set of normalised plane 2 pixel permittivity values

rem capfile1$ is the name of the plane1 capacitance data file
rem capfile2$ is the name of the plane2 capacitance data file

rem C1(maxframes,M) is the array which holds the C values read from the plane 1 data file
rem C2(maxframes,M) is the array which holds the C values read from the plane 2 data file


global framenumber,nframes,nframes2             'Declares Global variable
global smatrix$,capfile1$, capfile2$,geom32in$  'In Liberty Basic, all global variables are valid and have the same values
global position,ndatastart,nrecords             'in the main program and all subprograms
global p,map,M,P,CP,E                           'Note that all arrays are global by definition in Liberty Basic
global colourscale,imagegain
global K1,Kr,NCONC,imagegain$,NCONC$
global imagegain,NCONC,Cmap,colourscale
global OS,avcon,avcon1,avcon2, planes$
global frameskip,framenumber
global inigain$,ininconc$,inikr$,inicmap$
global iniframeskip$,iniplanes$,inielectrodes$
global inicapfile1$,inicapfile2$,capfile1$,capfile2$


rem ***** start of input data section

E = 8                                           'E is number of electrodes
p=32                                            'p is the number of pixels in the horizontal and vertical grid directions
P=p^2                                           'P is the total number of pixels in the square grid
CP=812                                          'CP is the number of pixels inside the circular boundary
maxframes = 1000                                'maxframes is maximum number of capacitance data frames to be input
K1=1                                            'K1 is the low (relative) permittivity value
geom32in$= "geom32.asm"                         'Geometry file details


rem define default parameters for ini file      'Note these values are overwritten when ini file is read by "readini"
inigain$ = "1"
ininconc$ = "3"
inikr$ = "1"
inicmap$ = "2"
iniframeskip$ = "5"
iniplanes$ = "1"
inielectrodes$ = "8"
inicapfile1$ = "c:\imagecon\8golf1.anc"
inicapfile2$ = "c:\imagecon\8golf2.anc"


call readini                                    'calls subroutine "readini" which reads default paremets from ini file


rem ***** end of data input section


dim K1(p,p),K2(p,p)                             'Set dimensions of plane1 and plane2 permittivity arrays
dim frame$(maxframes)                           'Set dimensions of capacitance frame number array
dim geom(p,p)                                   'Set dimensions of sensitivity map geometry array
dim RC(1),GC(1),BC(1)                           'Set dimensions of colour component arrays (RGB)
dim smat$(20),options$(20)                      'Set dimensions of inverse sensitivity matrix options array


nomainwin                                       'There is no main window

smat$(0)= "lbp12_32t.asm"                       'The inverse sensitivity matrix options are defined here
smat$(1)= "tik12_32_10t.asm"                    'lbp means Linear Back Projection
smat$(2)= "tik12_32_1t.asm"                     'tik means Tikhonov

smat$(3)= "lbp8_32t.asm"                        'the final t in the name indicates an inverse (transpose) matrix
smat$(4)= "tik8_32_10t.asm"
smat$(5)= "tik8_32_1t.asm"

smat$(6)= "lbp6_32t.asm"
smat$(7)= "tik6_32_10t.asm"
smat$(8)= "tik6_32_1t.asm"




rem **************** Start of main program **********************


call backwin                                        'calls subprogram "backwin" which opens background window
                                                    'which fills the dsiplay screen with a black background

call fileselwin                                     'calls subprogram "fileselwin" which opens a window
                                                    'for data file selection

call parawin                                        'calls subprogram "parawin" which opens a window for
                                                    'display parameter data entry

call textwin                                        'calls subprogram "textwin" which opens a window for
                                                    'text output


rem ***** Input file and parameter data

call filesel                                        'calls subprogram "filesel" which allows data entry into
                                                    'File Selection Window


dim C1(maxframes,M),C2(maxframes,M)                 'Sets dimensions of Capacitance data frame arrays and inverse
dim ST(P,M),C1m(maxframes,M),Ca$(M)                 'sensitivity matrix arrays once the required parameters have been
                                                    'set by the user in the "File selection" window

call paraminput                                     'calls data input subprogram "paraminput"

call kimagewin1                                     'calls subprogram "kimagewin1" which opens plane 1

if planes$="2" then                                 'Check whether there is a second plane of capacitance data.
call kimagewin2                                     'calls subprogram "kimagewin2" which opens plane 2
end if                                              'permittivity image window


rem ***** Read capacitance data files

call readcaps1                                      'calls subprogram "readcaps1" which reads the plane 1
                                                    'capacitance data file to array C1(1000,M)

if planes$="2" then                                 'Check whether there is a second plane of capacitance data.
call readcaps2                                      'If so, calls subprogram "readcaps2" which reads the plane 2
                                                    'capacitance data file to array C2(1000,M)
end if


call concmodel                                      'calls subprogram "concmodel" which implements the
                                                    'permittivity/concentration model by modifying the
                                                    'measured capacitances


rem ***** Read and plot sensitivity map geometry file

call readgeomfile                                   'calls subprogram "readgeomfile" which reads the sensitivity
                                                    'map geometry file which defines the active pixels



rem ***** Read the inverse sensitivity matrix file

open smatrix$ for input as #smat                    'opens the inverse sensitivity matrix file

call readsenscoeffs                                 'calls subprogram "readsenscoeffs" which reads the
                                                    'sensitivity coefficients from the inverse sensitivity matrix

close #smat                                         'closes the inverse sensitivity matrix file




rem ***** Start of the image reconstruction and plotting loop


framenumber = 0                                     'sets the initial framenumber to zero

for framenumber = 1 to nframes step frameskip       'nframes is the number of capacitance data frames
                                                    'read by subprogram "readcaps1"

call imagecon                                       'calls subprogram imagecon and constructs the permittivity images

call plotimage1                                     'calls subprogram "plotimage1" which plots the plane 1 permittiviy image


if planes$="2" then                                 'check whether there is a second plane of capacitance data

call plotimage2                                     'If so, call subprogram "plotimage2" which plots the
                                                    'Plane 2 permittiviy image

end if


scan                                                'Check for user input


goto [Pauseskip]

[Pause]

wait

[Resume]

[Pauseskip]



next framenumber                                    'keep looping until all the capacitance data frames have been plotted






rem ***** End of plotting loop



#paramin.exit "!setfocus"                           'set "Exit" button in the Parameter window to receive input focus

wait                                                'Wait for a button to be clicked


[exit]                                              'Go here to exit main program if "Exit" button clicked


call writeini                                       'Writes the current setup parameters to the ini file.


close #kimg1                                        'Close all the windows before exiting the main program
close #txtWin
close #backwin
close #paramin
close #filesel


if planes$="2" then
close #kimg2
end if



end

rem **************** End of main program **********************





rem                         **** Subprograms start here ****


rem *****

rem                                                 'subprogram "backwin" creates a black background window
                                                    'which fills the display screen
sub backwin
UpperLeftX=1                                        'Defines location of top LHS corner of window
UpperLeftY=1                                        'Note that 0,0 gives a "floating window"

WindowWidth  = DisplayWidth                         'sets window width to that of monitor display
WindowHeight = DisplayHeight                        'sets window height to that of monitor display

open "Image Reconstruction" for graphics_nsb_nf as #backwin
print #backwin, "fill black"                        'fills the window with black pixels
end sub





rem *****

rem                                                 'subprogram "fileselwin" creates the file selection window
sub fileselwin
UpperLeftX=30
UpperLeftY=30
WindowWidth = 430 : WindowHeight = 280


options$(0) = smat$(0)                              'Inverse sensitivity matrix file
options$(1) = smat$(1)                              'options are defined here
options$(2) = smat$(2)

options$(3) = smat$(3)
options$(4) = smat$(4)
options$(5) = smat$(5)

options$(6) = smat$(6)
options$(7) = smat$(7)
options$(8) = smat$(8)


rem  The size and location of the text boxes and buttons in the file selection window are defined here

statictext #filesel.s1, "Inverse Sensitivity Matrix:", 240, 5, 180, 30
listbox #filesel.list, options$(), [Filesselected], 240, 30, 180, 90

statictext #filesel.s2, "Plane1 capacitance file", 5, 5, 150, 25
textbox #filesel.capfile1, 5, 30, 150, 25

statictext #filesel.s3, "Plane2 capacitance file", 5, 70, 150, 25
textbox #filesel.capfile2, 5, 95, 150, 25

statictext #filesel.s4, "Number of planes", 5, 140, 110, 25
textbox #filesel.planes, 120, 140, 30, 25

statictext #filesel.s5, "Number of Electrodes", 240, 140, 130, 25
textbox #filesel.electrodes, 380, 140, 30, 25

statictext #filesel.output1, "Select Data files/parameters then press Continue button", 5, 170, 380, 20

button #filesel.b1, "Sel.P1",[cap1],UL, 170, 30, 50, 25
button #filesel.b2, "Sel.P2",[cap2],UL, 170, 95, 50, 25
button #filesel.cont, "Continue", [Filesselected], UL,320, 220, 80, 25
button #filesel.reset, "Reset", [Reset], UL,5, 220, 80, 25

open "Select input data files" for window as #filesel   'Opens the file selection window
#filesel "font ms_sans_serif 10"                        'set fonts for all controls

#filesel.planes iniplanes$                              'Default values defined here
#filesel.electrodes inielectrodes$

capfile1$ = inicapfile1$                                'sets default plane1 capacitance data file
capfile2$ = inicapfile2$                                'sets default plane2 capacitance data file

print #filesel.capfile1, capfile1$                      'prints default capacitance file names in text boxes
print #filesel.capfile2, capfile2$

end sub



rem *****

rem             'subprogram "parawin" creates parameter input window
sub parawin
UpperLeftX = 30
UpperLeftY = 325
WindowWidth = 430 : WindowHeight = 200

rem  The size and location of the text boxes and buttons in the parameter input window are defined here

statictext #paramin.s1, "Image Gain:", 5, 5, 80, 20
textbox #paramin.gain, 90, 5, 30, 25

statictext #paramin.s2, "N CONC:", 5, 35, 80, 20
textbox #paramin.NCONC, 90, 35, 30, 25

statictext #paramin.s3, "KR:", 150, 35, 80, 20
textbox #paramin.KR, 240, 35, 30, 25

statictext #paramin.s4, "Colour scale:", 150, 5, 80, 20
textbox #paramin.Cmap, 240, 5, 30, 25

statictext #paramin.output1, "Set values and click OK to confirm", 5, 100, 280, 20

statictext #paramin.s5, "Plot every ", 5, 65, 80, 20
textbox #paramin.frameskip, 90, 65, 30, 20
statictext #paramin.s6, " frames", 130, 65, 80, 20

statictext #paramin.output3, " ", 5, 125, 380, 20
statictext #paramin.output4, " ", 5, 145, 380, 20

button #paramin.ok, "OK",[OK],UL, 300, 5, 80, 25
button #paramin.cont, "Plot Images",[cont],UL, 300, 35, 80, 25
button #paramin.Pause, "Pause",[Pause],UL, 200, 65, 80, 25
button #paramin.Resume, "Resume",[Resume],UL, 300, 65, 80, 25
button #paramin.exit, "Exit",[exit],UL, 300, 95, 80, 25

Open "Input parameters window" for Window as #paramin   'Opens the parameter input window

#paramin "font ms_sans_serif 10"                        'set fonts for all controls


#paramin.gain inigain$                                  'set default value of image gain in textbox
#paramin.NCONC ininconc$                                'set default value of NCONC in textbox
#paramin.KR inikr$                                      'set default value of Kr in textbox
#paramin.Cmap inicmap$                                  'set default value of colour map in textbox
#paramin.frameskip iniframeskip$                        'set default frameskip parameter

end sub



rem *****

rem             'subprogram "textwin" creates text output window

sub textwin
UpperLeftX=30
UpperLeftY=545
WindowWidth=430:WindowHeight=200
open "Input Parameters" for text as #txtWin     'Opens the input parameters window
end sub



rem *****

rem                 'subprogram "kimagewin1" creates the permittivity image window for plane 1
sub kimagewin1
UpperLeftX=500
UpperLeftY=30
WindowWidth=500:WindowHeight=360
open "Plane 1 Permittivity Image" for graphics_nsb_nf as #kimg1     'opens the plane 1 permittivity image window
print #kimg1, "fill darkgray"                   'fills the window with dark grey pixels
end sub


rem *****

rem                 'subprogram "kimagewin2" creates the permittivity image window for plane 2

sub kimagewin2
UpperLeftX=500
UpperLeftY=400
WindowWidth=500:WindowHeight=360
open "Plane 2 Permittivity Image" for graphics_nsb_nf as #kimg2     'opens the plane 2 permittivity image window
print #kimg2, "fill darkgray"                    'fills the window with dark grey pixels
end sub



rem *****

rem     'subprogram "filesel" selects the capacitance and sensitivity matrix files to be used for image reconstruction

sub filesel

[Reset]                                     'Go here if Reset button clicked

#filesel.capfile1 "!setfocus"

wait                                        'Wait for a button to be pressed

[cap1]                                      'Go here if "Sel.P1" button clicked (selects Plane1 data file)


rem Plane 1 capacitance data file browse/selection dialogue follows

open "Select Plane 1 capacitance data file" for window as #capsel1
#capsel1 "font ms_sans_serif 10"
filedialog "Select Plane 1 capacitance file", "*.anc", filename$
if filename$<>"" then
capfile1$=filename$
print #txtWin, "Plane1 capacitance file is: ";capfile1$
print #filesel.capfile1, capfile1$
else
print #filesel.capfile1,  "No file chosen!"
end if
close #capsel1

wait                                        'Wait for a button to be pressed

[cap2]                                      'Go here if "Sel.P2" button clicked  (selects Plane 2 data file)

rem Plane 2 capacitance data file browse/selection dialogue follows

open "Select Plane 2 capacitance data file" for window as #capsel2
#capsel2 "font ms_sans_serif 10"
filedialog "Select Plane 2 capacitance file", "*.anc", filename$
if filename$<>"" then
capfile2$=filename$
print #txtWin, "Plane2 capacitance file is: ";capfile2$
print #filesel.capfile2, capfile2$
else
print #filesel.capfile2,  "No file chosen!"
end if
close #capsel2

wait                                                'Wait for a button to be pressed



[Filesselected]                                     'Go here if Continue button pressed



print #filesel.list, "selection? smatrix$"          'copy selected inverse sensitivity matrix file name to smatrix$

#filesel.electrodes "!contents? electrodes$"        'get number of electrodes from textbox:

#filesel.planes "!contents? planes$"                'get number of planes from textbox:

E= val(electrodes$)                                 'E is number of electrodes

M = E*(E-1)/2                                       'M is number of unique capacitance-pair measurements




print #txtWin, "Capacitance file1 is "; capfile1$   'Print following data to textwindow

print #txtWin, "Capacitance file2 is "; capfile2$

print #txtWin, "Inverse sensitivity matrix file is "; smatrix$ 

print #txtWin, "Number of electrodes = ";E

print #txtWin, "Number of unique capacitance pairs = ";M



iniplanes$ = planes$
inielectrodes$ = electrodes$
inicapfile1$ = capfile1$
inicapfile2$ = capfile2$



end sub





rem *****

rem     'subprogram "paraminput" sets the parameters to be used for image reconstruction

sub paraminput                                          'inputs paremeters to Parameter window

[Reset]

#paramin "trapclose [quit]"                             'trap the close event

#paramin.gain "!setfocus"                               'set textbox gain box to receive input focus

Wait                                                    'wait for a button to be pressed

[quit]
goto [subend]

[cont]                                                  'go here when "continue" button clicked
goto [subend]                                           'exit subprogram

[OK]                                                    'go here when "OK" button clicked

#paramin.gain "!contents? imagegain$"                   'get value of image gain from textbox:
#paramin.NCONC "!contents? NCONC$"                      'get value of NCONC from textbox:
#paramin.KR "!contents? KR$"                            'get value of KR from textbox:
#paramin.Cmap "!contents? Cmap$"                        'get value of colourscale from textbox:
#paramin.frameskip "!contents? frameskip$"              'get number of frames to skip from textbox:

frameskip = val(frameskip$)
print #txtWin,"Skip every ";frameskip;" frames"


imagegain = val(imagegain$)
NCONC=val(NCONC$)
Kr=val(KR$)
colourscale=val(Cmap$)

msg$ = "Click Plot Images button to continue,"
#paramin.output3 msg$ 

msg$ = "then click Exit button at end of image sequence to quit program "
#paramin.output4 msg$ 



wait                                                    'wait for a button to be pressed

[subend]                                                'go here when "Continue" button clicked

inigain$ = imagegain$
ininconc$ = NCONC$
inikr$ = KR$
inicmap$ = Cmap$
iniframeskip$ = frameskip$




end sub



rem *****

rem                 'subprogram "readcaps1" reads the plane 1 normalised capacitance data file
sub readcaps1



open capfile1$ for input as #caps1              'opens normalised capacitance file

rem read header section

for linecount = 1 to 9
line input #caps1,text$ 
header$(linecount)=text$
print #txtWin,header$(linecount)
next linecount


rem                                 'read subsequent frames of capacitance data

nf=0                                            'set intial frame count to zero

Do while EOF(#caps1)=0

line input #caps1,text$                         'read blank line

nf=nf+1                                         'increase frame count by one and read first frame of data

rem read frame header line
line input #caps1,text$                         'reads first line of capacitance data
rem print text$

eoftest$=word$(text$,1)                         'test for ##EOF text and end read file peration id found
if eoftest$ = "##EOF" then
exit do
end if

frame$(nf) = word$(text$,3)                     'extract frame number


m=1                                             'm is the capacitance measurement number (m = 1-M)
nseq =(E-1)                                     'nseq is the number of capa measurements in first line of data
for nline = 1 to nseq                           'nline is number of lines of data in frame
line input #caps1,text$                         'reads contents of data line to character string text$
for ndat = 1 to nseq                            'ndat is number of capa measurements in each line of data
Ca$(m) = word$(text$,ndat)                      'breaks text$ down into individual capacitance data measurements Ca$(m)
C1(nf,m)=val(Ca$(m))                            'converts CaS(m) ascii to C1(nf,m) numerical data
m=m+1
next ndat
nseq=nseq-1                                     'decreases nseq by one for next line of data
next nline

loop

close #caps1
nframes=nf-1
print #txtWin,"total number of capacitance dataframes is ";nframes
end sub


rem *****

rem                     subprogram "readcaps2" reads the plane 2 normalised capacitance data file
sub readcaps2

open capfile2$ for input as #caps2                  'opens normalised capacitance file

rem read header section

for linecount = 1 to 9
line input #caps2,text$ 
header$(linecount)=text$
print #txtWin,header$(linecount)
next linecount

rem read subsequent frames of capacitance data

nf=0                                            'set intial frame count to zero

Do while EOF(#caps2)=0

line input #caps2,text$                         'read blank line

nf=nf+1                                         'increase frame count by one and read first frame of data

rem read frame header line
line input #caps2,text$                         'reads first line of capacitance data

eoftest$=word$(text$,1)                         'test for ##EOF text and end read file peration id found
if eoftest$ = "##EOF" then
exit do
end if

frame$(nf) = word$(text$,3)                     'extracts frame number

m=1                                             'm is the capacitance measurement number (m = 1-M)
nseq =(E-1)                                     'nseq is the number of capa measurements in first line of data
for nline = 1 to nseq                           'nline is number of lines of data in frame
line input #caps2,text$                         'reads contents of data line to character string text$
for ndat = 1 to nseq                            'ndat is number of capa measurements in each line of data
Ca$(m) = word$(text$,ndat)                      'breaks text$ down into individual capacitance data measurements Ca$(m)
C2(nf,m)=val(Ca$(m))                            'converts CaS(m) ascii to C1(nf,m) numerical data
m=m+1
next ndat
nseq=nseq-1                                     'decreases nseq by one for next line of data
next nline

loop

close #caps2
nframes=nf-1
print #txtWin,"total number of capacitance dataframes is ";nframes
end sub


rem *****

rem             subprogram "concmodel" implements the permittivity/concentration model

sub concmodel                                   'Applies concentration model to capacitances
for nf = 1 to nframes                           'Reads capacitance data for each frame and modifies it
for mf= 1 to M

if Kr <> 1 then                                 'check to avoid potential problem for Kr = 1 which gives den = 0
K1n=C1(nf,mf)
K2n=C2(nf,mf)
K2= K1*Kr

K1m= K1n*(K2-K1) + K1
num1= (K1m - K1)*(K2 + K1*(NCONC -1))
den1 = K1*NCONC*(K2-K1) + (K1m - K1)*(K2-K1)
x1=num1/den1
C1(nf,mf)= x1

K2m= K2n*(K2-K1) + K1
num2= (K2m - K1)*(K2 + K1*(NCONC -1))
den2 = K1*NCONC*(K2-K1) + (K2m - K1)*(K2-K1)
x2=num2/den2
C2(nf,mf)= x2

end if                                          'if Kr = 0 don't modify the capacitance values
rem print nf,mf,x
next mf
next nf
end sub



rem *****

rem                         'subprogram "readgeomfile" reads the inverse sensitivity matrix geometry file
sub readgeomfile
open geom32in$ for input as #g32in
for i=1 to 32
for j=1 to 32
input #g32in,geom(i,j)
rem print geom(i,j);" ";
next j
rem print
next i
close#g32in
end sub



rem *****

rem                 'subprogram "readsenscoeffs" reads the inverse sensitivity matrix file
sub readsenscoeffs
rem Read the sensitivity coefficients
for n=1 to P
for m=1 to M
input #smat,coeff                          'Reads the sensitivity coefficients to the matrix ST(P,M)
ST(n,m)=coeff
next m
next n
end sub



rem *****
rem                 'subprogram "imagecon" constructs the permittivity image
sub imagecon
rem constructs the permittivity images and calculate average concentrations %
n=1
conc1=0
conc2=0
for i=1 to 32
for j=1 to 32
KS1=0
KS2=0
for m=1 to M
KS1=ST(n,m)*C1(framenumber,m)+KS1

if planes$="2" then
KS2=ST(n,m)*C2(framenumber,m)+KS2
else
KS2=1
end if

next m
K1(i,j)=KS1
K2(i,j)=KS2
n=n+1
conc1 = conc1+ K1(i,j)
conc2 = conc2+ K2(i,j)
next j
j=j-1
rem print i,K1(i,j),K2(i,j)
next i
rem print "image constructed"
avcon1 = 100*conc1/CP
avcon2 = 100*conc2/CP                                'calculates average concentration over pixels inside sensor
end sub




rem *****

rem                         'Subprogram "Plotimage1" plots the Plane 1 ECT image
sub plotimage1
print #kimg1, "down; size 1; color black"
print #kimg1, "backcolor black"

pw=10                                                   'p is dimension of image pixel expressed as number of window pixels
for i=1 to 32                                           'i is no of image pixels in X direction
i1=10*(i-1)
i2=i1+pw
for j=1 to 32                                           'j is no of image pixels in Y direction
j1=10*(j-1)
j2=j1+pw
x1=j1+10                                                'offsets image to right by x pixels
y1=i1+10                                                'offsets image down by y pixels
print #kimg1, "place ";x1;" ";y1                        'place pen at top LHS of next image pixel
x=x1+pw
y=y1+pw

pixcol = imagegain*K1(j,i)                              'Sets image gain. i,j reversed to correct direction of rotation
pc=10*pixcol

if colourscale = 1 then
call colourscale1 pc
else
call colourscale2 pc
end if

R=RC(1)
G=GC(1)
B=BC(1)


print #kimg1, "backcolor ";R;" ";G;" ";B                 'this is the correct format for defining graduated colour backgrounds

if geom(i,j)=0 then
print #kimg1, "backcolor black"
end if

print #kimg1, "boxfilled ";x;" ";y

next j
next i
print

rem create white background for concentration scale
for i=33 to 34                                          'i is no of image pixels in X direction
i1=10*(i-1)
i2=i1+pw
for j=6 to 26                                           'j is no of image pixels in Y direction (=200)
j1=10*(j-1)
j2=j1+pw
x1=i1+20
y1=j1+30
print #kimg1, "place ";x1;" ";y1                         'place pen at top LHS of next image pixel
x=x1+pw
y=y1+pw
print #kimg1, "backcolor white"
print #kimg1, "boxfilled ";x;" ";y
next j
next i

avconc1$ = using("###.##", avcon1)                      'truncates avcon1 to 2dp for printing

rem print text in permittivity window
avcon$ = using("###.##", avcon)
print #kimg1, "backcolor white"                         'Set back fill colour to white to make text visible

yo=180
print #kimg1, "place ";340;" ";25                       'locate pen above image
print #kimg1, "\Plane 1"                                'print Plane number
print #kimg1, "place ";410;" ";25                       'locate pen at upper right
print #kimg1, "\Frame ";frame$(framenumber)             'print frame number
print #kimg1, "place ";380;" ";150                      'locate pen at upper right
print #kimg1, "\Av conc. ";avconc1$;"%"                 'print average concentration
print #kimg1, "place ";340;" ";50                       'locate pen at lower right
print #kimg1, "\Conc %"                                 'print concentration
print #kimg1, "place ";340;" ";70                       'locate pen at upper right
print #kimg1, "\100 "                                   'print upper concentration limit (100)
print #kimg1, "place ";350;" ";310                      'locate pen at lower right
print #kimg1, "\0 "                                     'print lower concentration limit (0)
print #kimg1, "place ";380;" ";70+yo                    'locate pen at upper right
print #kimg1, "\Image gain= ";imagegain                 'print image gain
print #kimg1, "place ";380;" ";90+yo                    'locate pen at upper right
print #kimg1, "\N-Conc.= ";NCONC                        'print EMWS factor NCONC and permittivity ration Kr
print #kimg1, "place ";380;" ";110+yo                   'locate pen at upper right
print #kimg1, "\KR = ";Kr                               'print EMWS factor NCONC and p[ermittivity ration Kr
print #kimg1, "place ";380;" ";130+yo                   'locate pen at upper right
print #kimg1, "\Colour scale ";colourscale              'print colour scale

rem print concentration bar graph
x1=350
y1=280                                                  'line starts at horiz baseline
y2=y1-2*avcon1                                          'makes 100% avcon1 scan 200 pixels
print #kimg1, "color red"                               'sets pen colour to red
print #kimg1, "size 10"                                 'set pen width = 10 pixels wide
print #kimg1, "line ";x1;" ";y1;" ";x1;" ";y2

end sub



rem *****

rem                         'Subprogram "Plotimage2" plots the Plane 2 ECT image
sub plotimage2
print #kimg2, "down; size 1; color black"
print #kimg2, "backcolor black"

pw=10                                                   'p is dimension of image pixel expressed as number of window pixels
for i=1 to 32                                           'i is no of image pixels in X direction
i1=10*(i-1)
i2=i1+pw
for j=1 to 32                                           'j is no of image pixels in Y direction
j1=10*(j-1)
j2=j1+pw
x1=j1+10                                                'offsets image to right by x pixels
y1=i1+10                                                'offsets image down by y pixels
print #kimg2, "place ";x1;" ";y1                        'place pen at top LHS of next image pixel
x=x1+pw
y=y1+pw

pixcol = imagegain*K2(j,i)                              'Sets image gain. i,j reversed to correct direction of rotation
pc=10*pixcol

if colourscale = 1 then
call colourscale1 pc
else
call colourscale2 pc
end if

R=RC(1)
G=GC(1)
B=BC(1)

print #kimg2, "backcolor ";R;" ";G;" ";B                 'this is the correct format for defining graduated colur backgrounds

if geom(i,j)=0 then
print #kimg2, "backcolor black"
end if

print #kimg2, "boxfilled ";x;" ";y
next j
next i
print

rem create white background for concentration scale
for i=33 to 34                                          'i is no of image pixels in X direction
i1=10*(i-1)
i2=i1+pw
for j=6 to 26                                           'j is no of image pixels in Y direction (=200)
j1=10*(j-1)
j2=j1+pw
x1=i1+20
y1=j1+30
print #kimg2, "place ";x1;" ";y1                        'place pen at top LHS of next image pixel
x=x1+pw
y=y1+pw
print #kimg2, "backcolor white"
print #kimg2, "boxfilled ";x;" ";y
next j
next i

avconc2$ = using("###.##", avcon2)                      'truncates avcon2 to 2dp for printing

rem print text in permittivity window
avcon$ = using("###.##", avcon)
print #kimg2, "backcolor white"                         'Set back fill colour to white to make text visible

yo=180
print #kimg2, "place ";340;" ";25                       'locate pen above image
print #kimg2, "\Plane 2"                                'print Plane number
print #kimg2, "place ";410;" ";25                       'locate pen at upper right
print #kimg2, "\Frame ";frame$(framenumber)             'print frame number
print #kimg2, "place ";380;" ";150                      'locate pen at upper right
print #kimg2, "\Av conc. ";avconc2$;"%"                 'print average concentration
print #kimg2, "place ";340;" ";50                       'locate pen at lower right
print #kimg2, "\Conc %"                                 'print concentration
print #kimg2, "place ";340;" ";70                       'locate pen at upper right
print #kimg2, "\100 "                                   'print upper concentration limit (100)
print #kimg2, "place ";350;" ";310                      'locate pen at lower right
print #kimg2, "\0 "                                     'print lower concentration limit (0)
print #kimg2, "place ";380;" ";70+yo                    'locate pen at upper right
print #kimg2, "\Image gain= ";imagegain                 'print image gain
print #kimg2, "place ";380;" ";90+yo                    'locate pen at upper right
print #kimg2, "\N-Conc.= ";NCONC                        'print EMWS factor NCONC and permittivity ration Kr
print #kimg2, "place ";380;" ";110+yo                   'locate pen at upper right
print #kimg2, "\KR = ";Kr                               'print EMWS factor NCONC and p[ermittivity ration Kr
print #kimg2, "place ";380;" ";130+yo                   'locate pen at upper right
print #kimg2, "\Colour scale ";colourscale              'print colour scale

rem print concentration bar graph
x1=350
y1=280                                                  'line starts at horiz baseline
y2=y1-2*avcon2                                          'makes 100% avcon1 scan 200 pixels
print #kimg2, "color red"                               'sets pen colour to red
print #kimg2, "size 10"                                 'set pen width = 10 pixels wide
print #kimg2, "line ";x1;" ";y1;" ";x1;" ";y2

end sub

rem *****



rem *****
rem                 'subprogram "creategeom" creates an integer array of active pixel data
sub creategeom2
rem generate and print geom2 coefficients
rem geom 2 is a linear array containing integer values (1 or 0)
count2 = 1
for m=1 to 32
for n=1 to 32
geom2(count2)=INT(geom(m,n))
rem print geom2(count2);" ";
count2=count2+1
next n
rem print
next m
end sub


rem *****

rem                     'subprogram "colourscale1" defines a bipolar colour scale for the permittivity image
sub colourscale1 pc              'This is a bipolar colour scale, red is positive 100%, blue is negative 100%, green is zero
if pc>10 then                    'use saturated colour if result of m will cuase colour value to exceed 255
pc=10
end if

if pc<0 then                     'sets graduated colour scale, -ve is blue, zero is green, positive is red
R=0
B=ABS(pc)*255/10
G=(10-ABS(pc))*255/10
else
B=0
R=ABS(pc)*255/10
G=(10-ABS(pc))*255/10
end if

RC(1)=int(R)
GC(1)=int(G)
BC(1)=int(B)
end sub



rem *****

rem                     'subprogram "colourscale2" defines a monopolar colour scale for the permittivity image
sub colourscale2 pc             'This is a monopolar colour scale, blue is zero, green is 50%, red is 100%
pc=pc*2

if pc>20 then
pc=20
end if

if pc<0 then
pc=0
end if

if pc<10 then
R=0
B=ABS(10-pc)*255/10
G=ABS(pc)*255/10
else
B=0
R=ABS(pc-10)*255/10
G=ABS(20-pc)*255/10
end if

RC(1)=int(R)
GC(1)=int(G)
BC(1)=int(B)
end sub

rem *****
rem                         subprogram "pause" generates an approximate 1 second delay
sub pause
for npause= 1 to 250                       'Pause loop for approx 1 second
for mpause=1 to 1000
next mpause
next npause
end sub


rem *****

rem                     subroutine "readini" reads the default data from the .ini file
sub readini

open "inifile.txt" for input as #ini
[loop]
n=1
while eof(#ini) = 0
input #ini, param$
iniparam$(n) = param$
print n, iniparam$(n)
n=n+1
wend
close#ini

inigain$ = iniparam$(1)
ininconc$ = iniparam$(2)
inikr$ = iniparam$(3)
inicmap$ = iniparam$(4)
iniframeskip$ = iniparam$(5)
iniplanes$ = iniparam$(6)
inielectrodes$ = iniparam$(7)
inicapfile1$ = iniparam$(8)
inicapfile2$ = iniparam$(9)

rem print, inigain$
rem print, ininconc$
rem print, inikr$
rem print, inicmap$
rem print, iniframeskip$
rem print, iniplanes$
rem print, inielectrodes$
rem print, inicapfile1$
rem print, inicapfile2$

end sub


rem *****

rem                     subprogram writeini updates ini file (writes last used data to inifile)

sub writeini

open "inifile.txt" for output as #inifile

print #inifile, inigain$
print #inifile, ininconc$
print #inifile, inikr$
print #inifile, inicmap$
print #inifile, iniframeskip$
print #inifile, iniplanes$
print #inifile, inielectrodes$
print #inifile, inicapfile1$
print #inifile, inicapfile2$

close #inifile

end sub


